﻿using System;
using System.Linq;
using System.ComponentModel;
using System.Web.UI.WebControls;

using Framework;
using Framework.Data;

namespace Framework.Web.UI
{
    /// <summary>
    /// BoundFieldの機能拡張版
    /// </summary>
    public class LBoundField : System.Web.UI.WebControls.BoundField
    {
        /// <summary>
        /// マッピング名。未指定の場合はDataFieldと同じ
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string MappingName 
        {
            get { return _mappingName.IsEmpty() ? base.DataField : _mappingName; }
            set { _mappingName = value; } 
        }
        private string _mappingName;

        /// <summary>
        /// HTMLタグをそのまま表示 ＋ 改行コードを<br/>に変換する場合は true にします。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool ReplaceHTMLTag { get; set; }

        /// <summary>
        /// リストキー。
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ListKey { get; set; }

        /// <summary>
        /// 編集時のコントロールのデフォルト値
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string DefaultValue
        {
            get { return (string)ViewState["DefaultValue"]; }
            set { ViewState["DefaultValue"] = value; }
        }

        /// <summary>
        /// 編集時のコントロールが必須かどうか
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool IsRequired { get; set; }

        /// <summary>
        /// 編集時のコントロールの書式
        /// </summary>
        [Category("Custom")]
        [DefaultValue("NotSet")]
        public InputFormat InputFormat { get; set; }

        /// <summary>
        /// 編集時のコントロールの数値の範囲や正規表現を指定します。
        /// [1...9] ⇒1以上9以下  ...9] ⇒ 9以下
        /// (1...9) ⇒1より大きく9より小さい　(1.. ⇒１より大きい
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string InputFormatOption { get; set; }

        /// <summary>
        /// 編集時のコントロールのDecimalの精度を"15,6"のように指定します。
        /// 
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string InputDecimalPrecision { get; set; }

        /// <summary>
        /// 編集時のコントロールの最大文字数
        /// </summary>
        [Category("Custom")]
        [DefaultValue("-1")]
        public int MaxLength { get; set; }

        /// <summary>
        /// 編集時のコントロールのS-JISにした時のバイト数の最大値
        /// </summary>
        [Category("Custom")]
        [DefaultValue("-1")]
        public int MaxBytesBySJIS { get; set; }

        /// <summary>
        /// 編集時のコンロトールに適応させるスタイル
        /// </summary>
        public string style { get; set; }

        /// <summary>
        /// リストソースリクエストイベント。ListKeyが設定されたときに呼び出されます。
        /// </summary>
        public event ListSrcRequestEventHandler ListSrcRequest;

        /// <summary>
        /// 親GridViewの参照を返します。
        /// </summary>
        private LGridView Parent
        {
            get { return this.Control as LGridView; }
        }

        /// <summary>
        /// 編集状態のときのコントロールの種類を指定します。
        /// </summary>
        public EditControlType EditControlType { get; set; }

        /// <summary>
        /// テキスト値をキーにして、テキストをローカライズするかどうか？
        /// </summary>
        public bool LocalizeByText { get; set; }

        /// <summary>
        /// ヘッダに表示するテキストのテキストキー
        /// </summary>
        public string HeaderLTextKey { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LBoundField()
        {
            this.ReplaceHTMLTag = false;
            this.InputFormat = InputFormat.NotSet;
            this.MaxLength = -1;
            this.MaxBytesBySJIS = -1;
            this.EditControlType = EditControlType.Default;
            this.LocalizeByText = false;

            //デフォルトをfalseに変更
            this.HtmlEncode = false;
        }

        /// <summary>
        /// Initialize
        /// </summary>
        /// <param name="enableSorting"></param>
        /// <param name="control"></param>
        /// <returns></returns>
        public override bool Initialize(bool enableSorting, System.Web.UI.Control control)
        {
            //ローカライズ
            if (this.HeaderLTextKey.IsNotEmpty())
            {
                this.HeaderText = LTextMgr.GetText(this.HeaderLTextKey);
            }

            return base.Initialize(enableSorting, control);
        }

        /// <summary>
        /// InitializeCell
        /// </summary>
        /// <param name="cell"></param>
        /// <param name="cellType"></param>
        /// <param name="rowState"></param>
        /// <param name="rowIndex"></param>
        public override void InitializeCell(DataControlFieldCell cell, DataControlCellType cellType, DataControlRowState rowState, int rowIndex)
        {
            if ((cellType == DataControlCellType.DataCell && (rowState & DataControlRowState.Edit) != 0)
                || (cellType == DataControlCellType.Footer && this.Parent.ShowInsertFooter))
            {
                //編集行、または行追加用フッターの場合

                //※ここではコントロールの種類を変更するだけです。値のマッピングは自前で実装すること。
                //※ただし、GridViewPager を使う場合は、Edit状態になるときに自動でマッピングが行われます（たぶん。。。）

                if (base.ReadOnly)
                {
                    //列が読み取り専用の場合はリテラルにする。

                    MLiteral lit = new MLiteral();
                    lit.MappingName = this.MappingName;
                    cell.Controls.Add(lit);
                }
                else if (this.ListKey.IsNotEmpty() && this.EditControlType == EditControlType.Default)
                {
                    //リストキーが設定されている場合は、DropDownListにする。

                    var ddl = new MDropDownList();
                    ddl.ListKey = this.ListKey;
                    if (this.ListSrcRequest != null)
                    {
                        ddl.ListSrcRequest += new ListSrcRequestEventHandler((sender, e) =>
                        {
                            e.Argument = "Edit";
                            this.ListSrcRequest(sender, e);
                        });
                    }
                    ddl.MappingName = this.MappingName;
                    ddl.DefaultValue = this.DefaultValue;
                    ddl.LocalizeByText = this.LocalizeByText;
                    ddl.Attributes["style"] = this.style;

                    cell.Controls.Add(ddl);
                }
                else
                {
                    //その他の場合は、TextBox にする。

                    MTextBox txtBox = new MTextBox();
                    txtBox.MappingName = this.MappingName;
                    txtBox.DefaultValue = this.DefaultValue;
                    txtBox.IsRequired = this.IsRequired;
                    txtBox.InputFormat = this.InputFormat;
                    txtBox.InputFormatOption = this.InputFormatOption;
                    txtBox.InputDecimalPrecision = this.InputDecimalPrecision;
                    if (this.MaxLength >= 0) txtBox.MaxLength = this.MaxLength;
                    txtBox.MaxBytesBySJIS = this.MaxBytesBySJIS;
                    txtBox.Attributes["style"] = this.style;

                    cell.Controls.Add(txtBox);
                }
            }
            else
            {
                base.InitializeCell(cell, cellType, rowState, rowIndex);
            }
        }

        ListSrcDataSet _listSrc = null;

        /// <summary>
        /// FormatDataValue の override
        /// </summary>
        /// <param name="dataValue"></param>
        /// <param name="encode"></param>
        /// <returns></returns>
        protected override string FormatDataValue(object dataValue, bool encode)
        {
            //ListKeyによる変換
            if (this.ListKey.IsNotEmpty() && dataValue != null)
            {
                if (this.ListSrcRequest != null)
                {
                    if (_listSrc == null)
                    {
                        //リストソースのリクエスト
                        var e = new ListSrcRequestEventArgs()
                        {
                            ListKey = this.ListKey,
                            Argument = "Normal"
                        };
                        this.ListSrcRequest(this, e);

                        _listSrc = e.ListSrc;
                    }
                    var row = _listSrc.ListSrc.FirstOrDefault(t => t.Value == dataValue.ToString());
                    if (row != null)
                    {
                        dataValue = row.Text;
                    }
                }
                else
                {
                    dataValue = ListMgr.GetText(this.ListKey, dataValue.ToString());
                }

            }

            //テキストのローカライズ
            if (this.LocalizeByText == true && dataValue != null)
            {
                dataValue = LTextMgr.GetText(dataValue.ToString());
            }


            //入力値が画面上でそれっぽく表示されるように、HTMLタグの置き換えをおこなう。
            if (this.ReplaceHTMLTag)
            {
                string str = dataValue as string;
                if (str.IsNotEmpty())
                {
                    str = System.Web.HttpUtility.HtmlEncode(str);
                    str = str.Replace("\r\n", "<br/>");
                    dataValue = str;
                    encode = false;
                }
            }

            return base.FormatDataValue(dataValue, encode);
        }
    }

    /// <summary>
    /// 編集時のコントロールの種類
    /// </summary>
    public enum EditControlType
    {
        /// <summary>
        /// デフォルト
        /// </summary>
        Default,

        /// <summary>
        /// テキストボックス
        /// </summary>
        TextBox
    }
}
